"""
Advanced Python School
OOP and Design Patterns

Solution to City Graph Exercise

by: Bartosz Telenczuk, Niko Wilbert
"""

import sys

sys.path.append("../examples")

import graph
import extended_shortest_path as sp


class CityNode(graph.Node):
    """Node representing a city with a name."""

    def __init__(self, name):
        """A node that represents a city with a name.

        Args:
            name (str): Name of the city.
        """
        super().__init__()
        self.name = name

    def __str__(self):
        """Return the city name."""
        return self.name


class TransportationEdge(graph.Edge):
    """Base class for edges representing a transportation connection.

    It connects CityNode instances.
    """

    def __init__(self, tail, head, time, cost, description):
        """Edge that represents a transportation connection.

        Args:
            tail: Start city node of this connection.
            head: End city node of this connection.
            time: Time this connection takes.
            cost (float, int): Cost of taking this connection.
            description (str): String describing the mode of transportation, like 'train'.
        """
        super().__init__(tail, head)
        self.time = time
        self.cost = cost
        self.description = description

    def __str__(self):
        """Return a nice description of this connection, including city names."""
        return f"{str(self.tail)} to {str(self.head)} via {self.description}"


class SearchGraph(graph.Graph):

    def __init__(self, edges=None):
        super().__init__(edges)
        self.search_algorithm = sp.SearchAlgorithm()

    def shortest_path(self, first, last):
        """Calculate the shortest path between *first* and *last*.

        Args:
            first: Start node
            last: End node

        Returns:

        """
        return self.search_algorithm.find(first, last)


if __name__ == '__main__':
    berlin = CityNode("Berlin")
    hamburg = CityNode("Hamburg")
    munich = CityNode("Munich")
    cologne = CityNode("Cologne")
    cities = [berlin, hamburg, munich]
    connections = [
        TransportationEdge(berlin, hamburg, 50, 100, "plane"),
        TransportationEdge(berlin, hamburg, 120, 50, "train"),
        TransportationEdge(berlin, munich, 40, 200, "plane"),
        TransportationEdge(munich, cologne, 40, 150, "plane"),
        TransportationEdge(hamburg, cologne, 30, 150, "plane"),
        TransportationEdge(hamburg, cologne, 240, 100, "train"),
    ]
    city_graph = SearchGraph(connections)
    city_graph.search_algorithm = sp.FastestPath()
    path, time = city_graph.shortest_path(berlin, cologne)
    print(f"fastest: {time:d} min")
    for edge in path:
        print(edge)
    print()
    city_graph.search_algorithm = sp.CheapestPath()
    path, time = city_graph.shortest_path(berlin, cologne)
    print(f"cheapest: {time:d} Euros")
    for edge in path:
        print(edge)
