""" logging with different formats for lower levels

- messages with loglevel >= ERROR are written to stderr
- all messages are stored in `mylog.txt`
  - messages with level >= ERROR store extra variables
  - messages below use a simpler format without extra data

"""

import logging

class LevelFilter(logging.Filter):
    """ filter all messages above a given level

    (minimal level can be defined using `Handler.setLevel()`)
    """
    def __init__(self, level):
        super().__init__()
        self._level = level

    def filter(self, record):
        if record.__dict__["levelno"] <= self._level:
            return 1
        else:
            return 0


# general setup + ouput to stderr
log = logging.Logger("splitlevel", level=logging.INFO)
log.addHandler(logging.StreamHandler())
log.handlers[-1].setLevel(logging.ERROR)

# log high-level messages to file (recreate)
# either use "a", or make sure to call the "w" handler first
log_high = logging.FileHandler("split.log", "a")
log_high.setLevel(logging.ERROR)
log_high.setFormatter(logging.Formatter("%(asctime)s -- %(levelname)s, %(var)s: %(message)s"))
log.addHandler(log_high)

# log low-level messages to same file
log_low = logging.FileHandler("split.log", "a")
log_low.addFilter(LevelFilter(logging.ERROR-1))  # apply this handler to everything below ERROR only
log_low.setFormatter(logging.Formatter("%(asctime)s -- %(levelname)s: %(message)s"))
log.addHandler(log_low)

# application code
log.info("This is just an info: %s", "foo")
log.error("This is bad", extra={"var": 42})

