#!/usr/bin/env python
import numpy as np
import matplotlib.pyplot as plt
from time import time

# Number of elements along the complex plane. Higher number gives higher resolution on the image.
# Don't go too crazy here. It will also allocate more and more memory. Too high values will lead to memory-swapping
# and then the time measurements make no sense anymore. You might want to check the memory consumption with
# linux-htop, mac-activity-monitor or the windows-task-manager
points_along_axis = 1500

# maximum iteration steps that are done per point in the complex plane to check whether it will diverge
# if you go above 255 you need to change the datatype (currently unit8) in the iterationmatrix as well!
maximum_iteration_steps = 200


# returns how many iterations you need to see that it diverges
# adapted from PM 2Ring via stackoverflow.com with changes to return iteration depth
# CC BY-SA 4.0
def slow_mandelbrot(points, maximum_iteration_steps):
    # define a iterations matrix for the elements in the complex plane, we will address them later with u, v
    iterations = np.ones([points, points], np.uint8) * maximum_iteration_steps
    # the real part
    x_values = np.linspace(-2, 2, points)
    # the imaginary part
    y_values = np.linspace(-2, 2, points)
    # each complex value can be identified with the indices u, v
    for u, x in enumerate(x_values):
        for v, y in enumerate(y_values):
            z = 0+0.j
            c = complex(x, y)
            for i in range(maximum_iteration_steps):
                z = z * z + c
                # as soon as we exceed 2 we know that it will diverge and we can stop for that element c.
                # We store at which iteration this occurred. That will give the color in the plot.
                if abs(z) > 2.0:
                    iterations[v, u] = i+1
                    break
    return iterations


# your implementation goes here
def faster_mandelbrot(points, maximum_iteration_steps):
    # just a dummy, your implementation goes here
    iterations = np.zeros((points, points), np.uint8)
    return iterations


print('calculating original image ...')
original_start_time = time()
slow_result = slow_mandelbrot(points_along_axis, maximum_iteration_steps)
original_stop_time = time()
original_timedelta = original_stop_time - original_start_time
print(f'original function calculation time {round(original_timedelta, 3)} sec')

print('calculating your optimized image ...')
start_time = time()
result = faster_mandelbrot(points_along_axis, maximum_iteration_steps)
stop_time = time()
optimized_timedelta = stop_time - start_time
print(f'your function calculation time {round(optimized_timedelta, 3)} sec')
print(f'speedup factor is {original_timedelta/optimized_timedelta:.2f}')


if not np.array_equal(slow_result, result):
    print('\n')
    print('! WARNING ! Your image is different than the original. But we will plot it anyways\n')


# Typically the images are plotted inverted. That is why we plot here maxiter-iterations
# This is just to make the image more familiar from popular science
f, (ax1, ax2) = plt.subplots(1, 2)
ax1.imshow(maximum_iteration_steps - slow_result, extent=[-2, 2, -2, 2], vmin=0, vmax=maximum_iteration_steps, cmap='nipy_spectral')
ax1.set_title('Original Slow method')
ax2.imshow(maximum_iteration_steps - result, extent=[-2, 2, -2, 2], vmin=0, vmax=maximum_iteration_steps, cmap='nipy_spectral')
ax2.set_title('your implementation')
print('Showing result. You can stop the script by closing the graph or ctrl-c')
plt.show()
