"""
OOP and Design Patterns

Solution to Decorator Pattern Exercise

by: Bartosz Telenczuk, Niko Wilbert, Nicola Chiapolini, Jonas Eschle
"""


class Beverage:
    """ Beverage interface

    Abstract base for beverages and ingredients.
    Defines the common functions of these classes.
    Putting `_name` here avoids implementing
    identical functions for both classes.

    """
    _cost = 0.00

    @property
    def _name(self):
        return self.__class__.__name__.lower()

    def __str__(self):
        return self._name

    @property
    def cost(self):
        return self._cost


class BaseBeverage(Beverage):
    """ Pure beverage

    functionally identical with beverage
    needed to distinguish base from ingredient
    (We could also have left `__str__` and `cost`
    empty in `Beverage` and implemented the
    specific behaviour here.)

    """


class Ingredient(Beverage):

    def __init__(self, beverage):
        self.base = beverage

    def __str__(self):
        if isinstance(self.base, BaseBeverage):
            sep = ", with "
        else:
            sep = " and "
        return f"{self.base}{sep}{self._name}"

    @property
    def cost(self):
        return self.base.cost + self._cost


class Coffee(BaseBeverage):
    _cost = 3.00


class Tea(BaseBeverage):
    _cost = 2.00


class Milk(Ingredient):
    _cost = 0.30


class Sugar(Ingredient):
    _cost = 0.20


class Cream(Ingredient):
    _cost = 0.50


class Sprinkles(Ingredient):
    _cost = 0.10


my_coffee = Sprinkles(Sugar(Milk(Coffee())))
print(f"Your {my_coffee} costs {my_coffee.cost:3.2f}.")
