# coding=utf-8

"""
Scientific Programming with Python
OOP and Design Patterns

Solution to Duck Exercise

by: Nicola Chiapolini
"""


class FlyingBehavior:
    """ default flying behavior """

    def __init__(self, duck):
        self.duck = duck

    def take_off(self):
        print("Runnig fast, flapping my wings, taking off from ", self.duck.position)

    def fly_to(self, destination):
        print("Now flying to", destination)
        self.duck.position = destination

    def land(self):
        print("Slowing down, extending legs, touch down in ", self.duck.position)


class NonFlyingBehavior(FlyingBehavior):
    """ flying behavior for ducks that can not fly """

    def take_off(self):
        print("It's not working :-(")

    def fly_to(self, destination):
        raise Exception("I'm not flying anywhere.")

    def land(self):
        print("That won't be necessary.")


class Duck:
    description = "Boring looking duck"

    """ Base class for all ducks """

    def __init__(self, name):
        self.name = name
        self.position = "Home"
        self.flying_behavior = FlyingBehavior(self)

    def set_flying_behaviour(self, behaviour):
        self.flying_behavior = behaviour(self)

    def take_off(self):
        self.flying_behavior.take_off()

    def fly_to(self, destination):
        self.flying_behavior.fly_to(destination)

    def land(self):
        self.flying_behavior.land()

    def quack(self):
        print("Quack!")

    def display(self):
        print("{desc}, called {name}".format(desc=self.description, name=self.name))


class RedheadDuck(Duck):
    description = "Duck with a read head"


class BlackDuck(Duck):
    description = "A really dark duck"


class RubberDuck(Duck):
    description = "Small yellow rubber duck"

    def __init__(self, name):
        super().__init__(name)
        self.flying_behavior = NonFlyingBehavior(self)

    def quack(self):
        print("Squeak!")


class DecoyDuck(Duck):
    def __init__(self, name):
        super().__init__(name)
        self.flying_behavior = NonFlyingBehavior(self)

    def quack(self):
        print("")


if __name__ == "__main__":
    ducks = []
    for name in ["Huey", "Dewey", "Louie"]:
        ducks.append(RedheadDuck(name))
    for name in ["Donald", "Daisy", "Fethry"]:
        ducks.append(Duck(name))
    ducks.append(BlackDuck("Scrooge McDuck"))
    ducks.append(RubberDuck("Rubby"))
    ducks.append(DecoyDuck("Louies"))
    for duck in ducks:
        duck.display()
    print()
    for duck in ducks:
        duck.take_off()
    print()
    unlucky_duck = ducks[3]
    print(unlucky_duck.name, "just broke its legs")
    unlucky_duck.set_flying_behaviour(NonFlyingBehavior)
    print()
    for duck in ducks:
        duck.take_off()
    print()
    travelling_duck = ducks[0]
    travelling_duck.fly_to("the lake")
    travelling_duck.land()
    print()
    for duck in ducks:
        print(duck.position)
