"""
Example solution for exercise of OOP.

written by Christian Elsasser for Scientific Programming with Python
modified by Andreas Weiden

This module contains all elements for vector classes.
"""

class Vector:
    """Base class for vectors
    """

    def __init__(self, numbers):
        """Initialize the vector.
        numbers - array of numbers
        """
        self._numbers  = numbers

    @property
    def dimension(self):
        """Get dimension of the vector"""
        return len(self._numbers)

    def __mul__(self, other):
        """Scalar multiplication
        other - vector with same dimension or scalar
        """
        if isinstance(other, Vector):
            if self.dimension != other.dimension:
                raise ValueError('dimension mismatch')
            numbers = [a * b for a, b in zip(self._numbers, other._numbers)]
            return sum(numbers)
        else:
            return Vector([other * a for a in self._numbers])

    def __add__(self, other):
        """Vector addition
        other - vector with same dimension
        """
        assert isinstance(other, Vector)
        if self.dimension != other.dimension:
            raise ValueError('dimension mismatch')
        numbers = [a + b for a, b in zip(self._numbers, other._numbers)]
        return Vector(numbers)

    def __str__(self):
        return "{" + ",".join(map(str, self._numbers)) + "}"

    @property
    def length(self):
        return sum(x**2 for x in self._numbers)**0.5

    @length.setter
    def length(self, length):
        old_length = self.length
        self._numbers = [x * length / old_length for x in self._numbers]

    @length.deleter
    def length(self):
        self.length = 0

class Vector3D(Vector):
    def __init__(self, numbers):
        """Initialize the vector.
        [x, y, z] - three dimensions
        """
        assert len(numbers) == 3
        super().__init__(numbers)

    @property
    def x(self):
        return self._numbers[0]
    @property
    def y(self):
        return self._numbers[1]
    @property
    def z(self):
        return self._numbers[2]

    def __matmul__(self, other):
        assert isinstance(other, Vector3D)
        x = self.y * other.z - self.z * other.y
        y = self.z * other.x - self.x * other.z
        z = self.x * other.y - self.y * other.x
        return Vector3D([x, y, z])

if __name__ == '__main__':
    v1 = Vector([2,3,4,6])
    v2 = Vector([3,2,1,4])
    v3 = v1 + v2
    print(str(v3))
    print(v1.length)
    v1.length = 5
    print(v1.length)
    del v1.length
    print(v1.length)
    v4 = Vector3D([1,2,3])
    v5 = Vector3D([4,5,6])
    print(v4 @ v5)


