#######################################################################
# This script compares the speed of the computation of a polynomial
# for different in-memory libraries: numpy and numexpr.
#
# Author: Francesc Alted
# Date: 2013-09-04
#######################################################################

from time import time
import numpy as np
import numexpr as ne

N = 10*1000*1000           # number of points to evaluate
x = np.linspace(-1, 1, N)  # vector x in range [-1, 1]

# The different expressions supported
expr = [
    ".25*x**3 + .75*x**2 - 1.5*x - 2",  # 0) the polynomial to compute
    "((.25*x + .75)*x - 1.5)*x - 2",    # 1) a computer-friendly polynomial
    "x",                                # 2) the identity function
    "sin(x)**2 + cos(x)**2",            # 3) a transcendental function
    ]

# Set here the index of the expression to compute
to_compute = 0

# Set here which library you want to use
what = "numpy"              # uses numpy for computations
#what = "numexpr"           # uses numexpr for computations


def compute():
    """Compute the polynomial using different methods."""
    expr_ = expr[to_compute]
    if what == "numpy":
        if "sin" in expr_:
            # Trick to allow numpy evaluate this
            expr_ = "np.sin(x)**2+np.cos(x)**2"
        elif expr_ == "x":
            # Trick to force a copy with NumPy
            y = x.copy()
        y = eval(expr_)
    else:
        ne.set_num_threads(1)			# force to use only 1 thread
        y = ne.evaluate(expr_)
    return y


if __name__ == '__main__':
    print("Computing:", expr[to_compute], "using", what, "with", N, "points")
    t0 = time()
    result = compute()
    ts = round(time() - t0, 3)
    print("result:", result)
    print("*** Time elapsed:", ts, "s")
