"""
Advanced Python School
OOP and Design Patterns

Solution to City Graph Exercise

by: Bartosz Telenczuk, Niko Wilbert
"""

import sys
sys.path.append("../examples")

import graph
import shortest_path as sp

class CityNode(graph.Node):
    """Node representing a city with a name."""
    
    def __init__(self, name):
        """Initialize the city node with a name."""
        super(CityNode, self).__init__()
        self.name = name
        
    def __str__(self):
        """Return the city name."""
        return self.name
        

class TransportationEdge(graph.Edge):
    """Base class for edges representing a transportation connection.
    
    It connects CityNode instances. 
    """
    
    def __init__(self, head, tail, time, cost, description):
        """Initialize the transportation node.
       
        head, tail -- End and start city node of this connection. 
        time -- Time this connection takes.
        cost -- Cost of taking this connection.
        description -- String describing the mode of transportation, like
            'train'.
        """
        super(TransportationEdge, self).__init__(head, tail)
        self.time = time
        self.cost = cost
        self.description = description
        
    def __str__(self):
        "Return a nice desciption of this connection, including city names."
        return "%s to %s via %s" % (str(self.tail), str(self.head),
                                    self.description)

class SearchGraph(graph.Graph):

    def __init__(self, edges=None):
        super(SearchGraph, self).__init__(edges)
        self.search_algorithm = sp.SearchAlgorithm()

    def shortest_path(self, first, last):
        return self.search_algorithm.find(first, last)

        
if __name__ == '__main__':
    berlin = CityNode("Berlin")
    hamburg = CityNode("Hamburg")
    munich = CityNode("Munich")
    cologne = CityNode("Cologne")
    cities = [berlin, hamburg, munich]
    connections = [
        TransportationEdge(hamburg, berlin, 50, 100, "plane"),
        TransportationEdge(hamburg, berlin, 120, 50, "train"),
        TransportationEdge(munich, berlin, 40, 200, "plane"),
        TransportationEdge(cologne, munich, 40, 150, "plane"),
        TransportationEdge(cologne, hamburg, 30, 150, "plane"),
        TransportationEdge(cologne, hamburg, 240, 100, "train"),
    ]
    city_graph = SearchGraph(connections)
    city_graph.search_algorithm = sp.FastestPath()
    path, time = city_graph.shortest_path(berlin, cologne)
    print "fastest: %d min" % time
    for edge in path:
        print edge
    print
    city_graph.search_algorithm = sp.CheapestPath()
    path, time = city_graph.shortest_path(berlin, cologne)
    print "cheapest: %d Euros" % time
    for edge in path:
        print edge
