#include "actions_staeckel.h"
#include "potential_factory.h"
#include "units.h"
#include "debug_utils.h"
#include "utils.h"
#include "math_core.h"

//Two i/o routines you might want to replace
void compress0(FILE*,float*,int);
bool decompress(FILE*,float*,int);

#include "star.h" //your star class
#include "starx.h" //your star class with AAvars and frequencies added
#include <iostream>
#include <fstream>
#include <cmath>
#include <omp.h>

#define UNIT 978


const units::InternalUnits unit(units::galactic_Myr);
//const double dim = unit.to_Kpc*unit.to_Kpc/unit.to_Myr;

potential::PtrPotential make_galpot(const char* params)
{
	const char* params_file="test_galpot_params.pot";
	std::ofstream out(params_file);
	out<<params;
	out.close();
	potential::PtrPotential gp = potential::readGalaxyPotential(params_file, unit);
	std::remove(params_file);
	if(gp.get()==NULL)
		std::cout<<"Potential not created\n";
	return gp;
}

const char* test_galpot_params =
// McMillan 2011 BestFitPotential.Tpot
				"3\n"
				"5.63482e+08 2.6771 0.1974 0 0\n"
				"2.51529e+08 2.6771 0.7050 0 0\n"
				"9.34513e+07 5.3542 0.04 4 0\n"
				"2\n"
				"9.49e+10    0.5  0  1.8  0.075   2.1\n"
				"1.85884e+07 1.0  1  3    14.2825 250.\n";

const char* M17_galpot_params =
// McMillan 2017 potential
			       "4\n"
			       "8.96e+08 2.5 0.3 0 0\n"
			       "1.83e+08 3.02 0.9 0 0\n"
			       "0.531+e08 7 .085 4 0\n"
			       "21.8+e08 1.5 .045 12 0\n"
			       "2\n"
			       "9.84e+10  0.5  0  1.8  0.075 2.1\n"
			       "0.854e+07 1.0  1  3    19.6 1e5\n";

const char* Piffl14_galpot_params =
// Phi fitted to RAVE by Piffl 2014
				   "3\n"
				   "5.707e8 2.68 0.2 0 0\n"
				   "2.51e8 2.68 0.7 0 0 \n"
				   "9.45e7 5.36 0.04 4 0\n"
				   "2\n"
				   "9.49e10 0.5 0 1.8 0.075 2.1\n"
				   "1.816e7 1 1 3 14.4 1e5\n";

//Before running this code, split the NSTAR stars over NPROC files 
//rvs_0,rvs_1,.. with equal numbers of sars in each, where NPROC
//is the numer of cores you have

#define NPROC 8
#define NSTAR 5596359

int main(void){
//Choose your Phi by changing argument of make_galpot()
	potential::PtrPotential pot = make_galpot(Piffl14_galpot_params);
	actions::ActionFinderAxisymFudge actfinder(pot);
	unsigned nstar[NPROC]={NPROC*0};
	unsigned ns1=NSTAR/NPROC; ns1++;
	double zmax=2.5;
	double Wsun=7.787,Rsun=8.3,zsun=.015;
#ifdef _OPENMP
#pragma omp parallel for schedule(dynamic,1)
#endif
	for(int file_no=0; file_no<NPROC; file_no++){//run over input files
		char ifname[20],ofname[20];
		sprintf(ifname,"../rvs_%d.cmp",file_no);//name of input file
		sprintf(ofname,"rvsx_%d.cmp",file_no);//name of output file
		FILE *ifile=fopen(ifname,"r"),*ofile=fopen(ofname,"w");
		if(ifile==NULL || ofile==NULL){
			printf("I can't open input and/or output file\n");
		}
		star s;
		while(s.getit(ifile)){
			//Impose criyteria for star selection
			if(fabs(s.y)>4. || fabs(s.R-Rsun)>0.5 || fabs(s.z)>zmax) continue;
			double speed=pow_2(s.Ug)+pow_2(s.Vg)+pow_2(s.Wh);
			if(speed>pow_2(500)) continue;
			coord::PosVelCar ic(s.x/unit.to_Kpc,s.y/unit.to_Kpc,(s.z+zsun)/unit.to_Kpc,
					    -s.Ug/unit.to_kms,s.Vg/unit.to_kms,(s.Wh+Wsun)/unit.to_kms);
			//comput actions and angles
			actions::Frequencies freq;
			actions::ActionAngles AAvars(actfinder.actionAngles(toPosVelCyl(ic),&freq));
			//store then output star with incremented data
			starx sx(s,AAvars,freq,unit);
			sx.putit(ofile);
			nstar[file_no]++;
		}
		fclose(ifile); fclose(ofile);
		printf("%s with %d stars completed\n",ofname,nstar[file_no]);
	}
}